<?php

namespace Modules\PlagiarismCheck;

use App\Lib\AiProvider;
use Modules\OpenAI\Contracts\Resources\PlagiarismContract;
use Modules\PlagiarismCheck\Traits\PlagiarismCheckApiTrait;
use Modules\PlagiarismCheck\Resources\PlagiarismDataProcessor;
use Modules\PlagiarismCheck\Responses\Plagiarism\GenerateResponse;
use Modules\PlagiarismCheck\Responses\Plagiarism\ReportResponse;
use Modules\OpenAI\Contracts\Responses\Plagiarism\GenerateResponseContract;
use Modules\OpenAI\Contracts\Responses\Plagiarism\ReportResponseContract;

use Modules\OpenAI\Contracts\Responses\AiDetector\AiDetectorGenerateResponseContract;
use Modules\OpenAI\Contracts\Responses\AiDetector\AiDetectorReportResponseContract;

use Modules\OpenAI\Contracts\Resources\AiDetectorContract;
use Modules\PlagiarismCheck\Resources\AiDetectorDataProcessor;

use Modules\PlagiarismCheck\Responses\AiDetector\GenerateResponse as AiDetectorGenerateResponse;
use Modules\PlagiarismCheck\Responses\AiDetector\ReportResponse as AiDetectorReportResponse;

class PlagiarismCheckProvider extends AiProvider implements PlagiarismContract, AiDetectorContract
{
    use PlagiarismCheckApiTrait;

    /**
     * Holds the processed data after it has been manipulated or transformed.
     * This property is typically used within the context of a class to store
     * data that has been modified or processed in some way.
     *
     * @var array Contains an array of data resulting from processing operations.
     */
    protected $processedData;

    public function description(): array
    {
        return [
            'title' => 'Plagiarism Check',
            'description' => __("The :x provider ensures originality by analyzing writing patterns and detecting external sources, fostering academic integrity and authenticity. It also includes AI detection capabilities to identify content generated by artificial intelligence.", ['x' => 'Plagiarism Check']),
            'image' => 'Modules/PlagiarismCheck/Resources/assets/image/logo.svg',
        ];
    }

    /**
     * Retrieve plagiarism options available for admin settings.
     *
     * @return array
     */
    public function plagiarismOptions(): array
    {
        return (new PlagiarismDataProcessor)->plagiarismOptions();
    }

    /**
     * Generate plagiarism and process the result.
     *
     * @return array
     */
    public function plagiarism(array $aiOptions): GenerateResponseContract
    {
        $this->processedData = (new PlagiarismDataProcessor)->plagiarismCheck($aiOptions);
        $url = moduleConfig('plagiarismcheck.PLAGIARISMCHECK.BASE_URL');
        return new GenerateResponse($this->generate($url));
    }

    /**
     * Retrieve the detailed plagiarism report for a given ID.
     *
     * @param int $id  The ID of the plagiarism check to retrieve the report for.
     * @return mixed  The plagiarism report data.
     */
    public function getPlagiarismReport(array $data): ReportResponseContract
    {
        $id = $data['id'];
        $state = $data['state'];

        $reportUrl = moduleConfig('plagiarismcheck.PLAGIARISMCHECK.BASE_URL') . '/report/' . $id;
        $statusUrl = moduleConfig('plagiarismcheck.PLAGIARISMCHECK.BASE_URL') . '/' . $id;

        if ($state == '5') {
            return new ReportResponse($this->getReport($reportUrl));
        }

        $maxAttempts = 10;
        $attempt = 0;

        do {
            $status = $this->checkStatus($statusUrl);
        
            // Stop the loop if the state is 5
            if (isset($status['data']['state']) && $status['data']['state'] == '5') {
                break;
            }

            // Stop the loop if the state is 4
            if (isset($status['data']['state']) && $status['data']['state'] == '4') {
                throw new \Exception(__('The content could not be checked due to an error that occurred during the plagiarism check. Please try again.'));
            }
        
            usleep(500000); // Sleep for 500 milliseconds
            $attempt++;
        
            // If max attempts reached, throw an exception
            if ($attempt >= $maxAttempts) {
                throw new \Exception(__('We were unable to retrieve the report at this time. Please try again.'));
            }
        
        } while (true);

        return new ReportResponse($this->getReport($reportUrl));
    }

    ## Start AI Detector

    /**
     * Retrieve plagiarism options available for admin settings.
     *
     * @return array
     */
    public function aiDetectorOptions(): array
    {
        return (new AiDetectorDataProcessor)->aiDetectorOptions();
    }

    /**
     * Generate plagiarism and process the result.
     *
     * @return array
     */
    public function aiDetector(array $aiOptions): AiDetectorGenerateResponseContract
    {
        $this->processedData = (new AiDetectorDataProcessor)->aiDetectorCheck($aiOptions);
        $url = moduleConfig('plagiarismcheck.PLAGIARISMCHECK.DETECTOR_URL');
        return new AiDetectorGenerateResponse($this->generate($url));
    }

    /**
     * Retrieve the detailed aidetector report for a given ID.
     *
     * @param int $id  The ID of the aidetector check to retrieve the report for.
     * @return mixed  The aidetector report data.
     */
    public function getAiDetectorReport(array $data): AiDetectorReportResponseContract
    {
        $id = $data['id'];
        $status = $data['status'];

        $statusUrl = moduleConfig('plagiarismcheck.PLAGIARISMCHECK.DETECTOR_URL') . $id . '/';

        if ($status == '5') {
            return new AiDetectorReportResponse($this->checkStatus($statusUrl));
        }

        $maxAttempts = 10;
        $attempt = 0;

        do {
            $status = $this->checkStatus($statusUrl);
        
            // Stop the loop if the status is 5
            if (isset($status['data']['status']) && $status['data']['status'] == '4') {
                break;
            }

            // Stop the loop if the status is 4
            if (isset($status['data']['status']) && $status['data']['status'] == '3') {
                throw new \Exception(__('The content could not be checked due to an error that occurred during the plagiarism check. Please try again.'));
            }
        
            usleep(500000); // Sleep for 500 milliseconds
            $attempt++;
        
            // If max attempts reached, throw an exception
            if ($attempt >= $maxAttempts) {
                throw new \Exception(__('We were unable to retrieve the report at this time. Please try again.'));
            }
        
        } while (true);

        return new AiDetectorReportResponse($this->checkStatus($statusUrl));
    }

    ## End AI Detector

    /**
     * Get the validation rules for a specific processor.
     * 
     * @param string $processor The name of the data processor class.
     * 
     * @return array Validation rules for the processor.
     */
    public function getValidationRules(string $processor): array
    {
        $processorClass = "Modules\\PlagiarismCheck\\Resources\\" . $processor;

        if (class_exists($processorClass)) {
            return (new $processorClass())->validationRules();
        }

        return [];
    }

    /**
     * Get the validation rules for a specific processor.
     * 
     * @param string $processor The name of the data processor class.
     * 
     * @return array Validation rules for the processor.
     */
    public function getCustomerValidationRules(string $processor): array
    {
        $processorClass = "Modules\\PlagiarismCheck\\Resources\\" . $processor;

        if (class_exists($processorClass)) {
            $rules = (new $processorClass())->customerValidation();
            return is_array($rules) ? $rules : [];
        }

        return [];
    }
}
